import asyncio
# import json
import subprocess
from . import port, routes, servers

class RequestManager:
    """
    A worker created with this class serves as the listener for an API from
    which the requests are then sent to a ThreadManager to schedule.
    """
    def __init__(self, n, job_queue, loop, host_addr, host_port,
        database_thread):
        self.job_queue = job_queue
        self.loop = loop
        self.host_port = host_port
        self.name = "request_man_" + str(n)
        self.host_addr = host_addr
        self.task = asyncio.ensure_future(self.start_server(), loop=self.loop)
        self.database = database_thread

    async def start_server(self):
        """
        Temporary backend for receiving requets
        """
        stop = False
        def handler(*args):
            servers.SimpleHTTPRequestHandler(self.job_queue, self.database, *args)
        httpd = servers.ThreadedHTTPServer((self.host_addr, self.host_port), handler)
        while not stop:
            httpd.handle_request()
            await asyncio.sleep(0.1)


class ThreadManager:
    """
    A worker created with this class serves to manage the creation of new
    threads to perform requested tasks.
    """
    def __init__(self, n, job_queue, data_queue, loop):
        self.job_queue = job_queue
        self.data_queue = data_queue
        self.loop = loop
        self.name = "thread_man_" + str(n)
        self.task = asyncio.ensure_future(self.run(), loop=self.loop)

    async def run_discovery_task(self, port_name):
        """
        This is done very badly and was mainly a hack to get it working.
        There is no reason to call a subprocess outside of the asyncio
        workflow other than that I could not get it to work initially and
        opted to spend my time elsewhere.
        The data queue task scheduling was spun out into here as async functions
        do not return standard values but rather a reference to the coroutine.
        This should be easily solvable but again, I have opted not to look into
        this right now.
        """
        target_port = 'config.ports.' + port_name # This is for OGCS only
        command = ["python",
            "/var/mnt/storage.nvlog/var/opt/python/lib/python3.6/site-packages/discovery/discover.py",
            "-p", target_port] # May the formatting Gods forgive my sins.
        p = subprocess.Popen(command, stdout=subprocess.PIPE)  # pylint: disable=consider-using-with
        while p.poll() is None:
            await asyncio.sleep(0.1)
        temp = p.stdout.read()
        data = temp.decode().rstrip().replace("None", "'None'")
        # data = json.loads(data)
        # This currently fails due to formatting issues but is the way to feed
        # discovery data back to the database modification pipeline.
        data_task = {port: data}
        self.data_queue.put_nowait(data_task)
        print("Task queued as {}".format(data_task))

    async def task_selection(self, task):
        """
        The only jobs that should be queued by this are port information
        changes. The output from this task (and input to the database queue)
        are solely in the format {port:data} in which the port data is a data
        struct corresponding to the desired change of port information.
        In short, the actual processing is done in this thread and the end
        result is then fed to the database manager to adjust the cached states.
        """
        for path, data in task.items():
            _, route, port_name = routes.check_route(path, "POST")
            output_data = routes.routes_POST[route]['handler'](port_name, data)
            # This is bad and needs to be fixed
            if output_data == "discover":
                await self.run_discovery_task(port_name)
                return
            data_task = {port: output_data}
            self.data_queue.put_nowait(data_task)
            print("Task queued as {}".format(data_task))

    def provision(self):
        """
        PALCEHOLDER
        This is the task which should perform the provisioning itself.
        """
        return

    async def run(self, delay=0.01):
        stop = False
        while not stop:
            task = await self.job_queue.get()
            await self.task_selection(task)
            self.job_queue.task_done()
            await asyncio.sleep(delay)


class DatabaseManager:
    """
    A worker created with this class serves to manage the reading and writing
    to and from the database. There should only be a single DatabaseManager
    instance running at any given time to avoid duplication and simultaneous
    read/writes causing corruption in the databse.
    Tasks are processed in the form a queue much like the other workers and
    can thus potentially bottleneck the system if usage inflates.
    """
    def __init__(self, data_queue, loop, database):
        self.data_queue = data_queue
        self.loop = loop
        self.name = "database_man"
        self.task = asyncio.ensure_future(self.run(), loop=self.loop)
        self.database = database
        self.lock = False

    def lock_db(self):
        self.lock = True

    def unlock_db(self):
        self.lock = False

    def port_exists(self, input_port_id):
        for ports in self.database:
            if ports.port_id == input_port_id:
                return True
        return False

    def add_port(self, port_id, port_data):
        new_port = port.Port(port_id=port_id)
        new_port.write_from_data(port_data)
        self.database.append(new_port)

    def change_port_data(self, port_data_map):
        """
        Temporary function for demonstration purposes.
        """
        for port_data in self.database:
            try:
                if port_data_map['port_id'] == port_data.port_id:
                    return port_data.write_from_data(port_data_map)
            except TypeError:
                pass # invalid data
        return False

    def get_port_data(self, input_port_id):
        while self.lock:
            pass # block
        for port_data in self.database:
            if input_port_id == port_data.port_id:
                return port_data.formatted_data()
        return {}

    async def run(self, delay=0.01):
        stop = False
        while not stop:
            job = await self.data_queue.get()
            await asyncio.sleep(delay)
            print("> Performing task: {}".format(job))
            while self.lock:
                await asyncio.sleep(0.1)
            self.lock_db()
            # Perform database modification(s) after locking
            for port_id, port_data in job.items():
                if not self.port_exists(port_id):
                    self.add_port(port_id, port_data)
                else:
                    self.change_port_data(port_data)
            self.unlock_db()
            self.data_queue.task_done()
