#!/usr/bin/env python3

"""
This library contains functions which allow for simple read/writing to and from
the data store in relation to client status for OpenVPN connections.
"""

import json
import os


# TODO The cache file needs to be hardcoded and possibly migrated to a better
# storage object/format


class Connection:
    """
    Connection: Models a client in openvpn aggregated against a remote session
    and against a specific lighthouse user, identified by the unix timestamp
    `time_unix` env var provided to MOST of the openvpn hooks.

    TODO state management
    There will be one entry for each OpenVPN connection
    Possible connection states:
       - Connected
       - Disconnected
       - Blocked (?)
    """

    # NOTE we _could_ just merge **kwargs into self.__dict__
    # explicitly assigning values is clearer and plays nicer with validation
    # ... though it doesn't validate anything currently

    def __init__(self,
                time_unix=None,
                username=None,
                client_id=None,
                client_ip=None,
                client_mac=None,
                session_id=None,
                lighthouse_ip=None,
                node_id=None,
                node_name=None,
                node_ip=None,
                node_port=None,
                bridge_iface=None,
                gretap_iface=None,
                logger_mute_flag=False,
                setup_commands=None,
                teardown_commands=None):
        self.time_unix = time_unix
        self.username = username
        self.client_id = client_id
        self.client_ip = client_ip
        self.client_mac = client_mac
        self.session_id = session_id
        self.lighthouse_ip = lighthouse_ip
        self.node_id = node_id
        self.node_name = node_name
        self.node_ip = node_ip
        self.node_port = node_port
        self.bridge_iface = bridge_iface
        self.gretap_iface = gretap_iface
        # Logger mute flag is used to prevent log spam when duplicate MAC addresses are attempted.
        self.logger_mute_flag = logger_mute_flag
        self.setup_commands = setup_commands
        self.teardown_commands = teardown_commands
        # TODO validation


class Cache:
    """
    Cache: Models a connection cache / data store, used as a mechanism to pass
    around state information between scripts and the API.

    This mechanism isn't "ideal", and there is at least one immediately viable
    alternative. This mechanism has, however, been deemed suitable for the
    foreseeable future, based on our understand of the planned scope of IP
    Access as a product.

    For visiblity's sake, this state could be managed in memory by a single
    process. There are many of different permutations of this architecture
    that could be applicable e.g. monolithic vs modular, with different ways
    and levels components could be separated on. Unless python proves
    unsuitable for some reason we probably won't even investigate any further,
    which fortunately seems very unlikely... at this stage.
    """

    def __init__(self, cache_dir):
        self._session_dir = f'{cache_dir}/sessions'
        self._address_dir = f'{cache_dir}/addresses'
        if not os.path.isdir(self._session_dir):
            raise ValueError(f'invalid cache dir: {self._session_dir}')
        if not os.path.isdir(self._address_dir):
            raise ValueError(f'invalid cache dir: {self._address_dir}')

    def reserve_address(self, mac_address):
        """
        reserve_address: Reserves the provided MAC address by creating
        a file titled with the MAC address.

        :param mac_address:
        """
        file = open(f"{self._address_dir}/{mac_address}", "w")  # pylint: disable=consider-using-with
        file.close()

    def cancel_reservation(self, mac_address):
        """
        cancel_reservation: Cancels a MAC address reservation by removing
        its associated address file.

        :param mac_address:
        """
        try:
            os.remove(f"{self._address_dir}/{mac_address}")
        except OSError:
            # File was probably already removed somehow.
            pass

    def check_reservation(self, mac_address):
        """
        check_reservation: Checks whether the provided MAC address has been
        reserved.

        :param mac_address:
        :return: True|False
        """
        return os.path.isfile(f"{self._address_dir}/{mac_address}")

    # TODO investigate json.JSONEncoder and json.JSONDecoder

    def load(self, time_unix):
        """
        load: Reads a Connection from the cache, looking up the provided
        time_unix, returning None if it doesn't exist.

        :param time_unix:
        :return: Connection|None
        """
        try:
            with open(f'{self._session_dir}/{time_unix}', 'r') as f:
                return Connection(**json.load(f))
        except FileNotFoundError:
            return None

    def store(self, connection):
        """
        store: Writes a Connection to the cache, saving it under it's
        time_unix.

        :param connection: Connection
        """
        with open(f'{self._session_dir}/{connection.time_unix}', 'w') as f:
            json.dump(connection.__dict__, f, allow_nan=False, separators=(',', ':'))
