"""
schema provides schemas for api model validation and documentation
"""

import marshmallow
from marshmallow import (
    fields,
    validate,
)
from netops import (
    automation_gateway as ag,
)
from netops.automation_gateway import (
    job,
)
from netops.grpcutil import (
    schema,
)

# field_mapping enumerates all the marshmallow field mappings for use by spec.py
field_mapping = {
    **schema.field_mapping,
}

# schema_mapping enumerates all the marshmallow schemas for use by spec.py
schema_mapping = {}


def _proto_schema(pkg, name):
    """_proto_schema wires up a new schema from a proto message, see the usage below"""
    return schema.register_message(schema_mapping, pkg.DESCRIPTOR.pool.FindMessageTypeByName(name))


Device = _proto_schema(ag, 'netops.automation_gateway.Device')

NmapScanJob = _proto_schema(job, 'netops.automation_gateway.job.NmapScanJob')

Condition = _proto_schema(ag, 'netops.automation_gateway.Condition')


class CreateAuthTokenRequest(marshmallow.Schema):
    request_type = fields.String(validate=validate.OneOf(['new', 'refresh']))
    node = fields.String()
    session = fields.String()
    url = fields.String()
    username = fields.String()
    password = fields.String()


class CreateAuthTokenResponse(marshmallow.Schema):
    token = fields.String()
    service_url = fields.String()
    service_name = fields.String()


class AuthTokenRequest(marshmallow.Schema):
    token = fields.String()


class ValidateAuthTokenResponse(marshmallow.Schema):
    valid: fields.Bool()


class RefreshAuthTokenResponse(marshmallow.Schema):
    token = fields.String()


class DevicesResponse(marshmallow.Schema):
    # devices are the sorted, limited, and potentially filtered result
    devices = fields.Nested(Device, many=True, required=True)
    # offset will be set if there is more data available see the query param
    offset = fields.String()


schema_mapping['AuthTokenRequest'] = AuthTokenRequest
schema_mapping['CreateAuthTokenRequest'] = CreateAuthTokenRequest
schema_mapping['CreateAuthTokenResponse'] = CreateAuthTokenResponse
schema_mapping['ValidateAuthTokenResponse'] = ValidateAuthTokenResponse
schema_mapping['RefreshAuthTokenResponse'] = RefreshAuthTokenResponse
schema_mapping['DevicesResponse'] = DevicesResponse


class DeviceResponse(marshmallow.Schema):
    device = fields.Nested(Device, required=True)


schema_mapping['DeviceResponse'] = DeviceResponse


class DeviceName(marshmallow.Schema):
    mac = fields.String(required=True)
    name = fields.String(required=True)


class DeviceNameList(marshmallow.Schema):
    devices_names = fields.List(fields.Nested(DeviceName), required=True, default=[])


schema_mapping['DeviceName'] = DeviceName
schema_mapping['DeviceNameList'] = DeviceNameList


class NmapScanGetJobResponse(marshmallow.Schema):
    job = fields.Nested(NmapScanJob, required=True)


class NmapScanDeleteJobResponse(marshmallow.Schema):
    job = fields.Nested(NmapScanJob, required=True)


schema_mapping['NmapScanGetJobResponse'] = NmapScanGetJobResponse
schema_mapping['NmapScanDeleteJobResponse'] = NmapScanDeleteJobResponse


class NodeStatus(marshmallow.Schema):
    conditions = fields.Nested(Condition, many=True, required=True, description="""Indicates the state of the node.
Ready indicates the state of the remote service + licensing (activation).
GuacReady indicates ready to create a new Guacamole session.
GuacActive indicates existing active guacamole session(s).""")


schema_mapping['NodeStatus'] = NodeStatus

class DeviceRedirect(marshmallow.Schema):
    mac = fields.String(required=True)
    redirect = fields.Bool(required=True)
    route = fields.String(required=True)


class DeviceRedirectList(marshmallow.Schema):
    devices_redirects = fields.List(fields.Nested(DeviceRedirect), required=True, default=[])


schema_mapping['DeviceRedirect'] = DeviceRedirect
schema_mapping['DeviceRedirectList'] = DeviceRedirectList
