############################################################################
# Copyright: Opengear 2021                                                 #
# Powershell script to deploy Lighthouse virtual machine to Hyper-V Server #
############################################################################

function pauseAndWait {
    Write-Host -NoNewLine "Press any key to continue..."
    $null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
}

# Get the ID and security principal of the current user account
$myWindowsID=[System.Security.Principal.WindowsIdentity]::GetCurrent()
$myWindowsPrincipal=new-object System.Security.Principal.WindowsPrincipal($myWindowsID)

# Get the security principal for the Administrator role
$adminRole=[System.Security.Principal.WindowsBuiltInRole]::Administrator

# Check to see if we are currently running "as Administrator"

if ($myWindowsPrincipal.IsInRole($adminRole)) {
    # We are running "as Administrator" - so change the title and background color to indicate this
    $Host.UI.RawUI.WindowTitle = $myInvocation.MyCommand.Definition + "(Elevated)"
    $Host.UI.RawUI.BackgroundColor = "DarkBlue"
    clear-host
} else {
    # We are not running "as Administrator" - so relaunch as administrator
    # Create a new process object that starts PowerShell
    $newProcess = new-object System.Diagnostics.ProcessStartInfo "PowerShell";
    # Specify the current script path and name as a parameter
    $newProcess.Arguments = $myInvocation.MyCommand.Definition;
    # Indicate that the process should be elevated
    $newProcess.Verb = "runas";

    # Start the new process
    [System.Diagnostics.Process]::Start($newProcess);
    # Exit from the current, unelevated, process
    exit
}

try {
    # Lighthouse image configuration
    $lh_Base_Memory = 8GB
    $lh_Generation = 1
    $lh_NetworkSwitch = "Lighthouse Switch"

    # Code that needs to be elevated here

    # Filename: lighthouse_virtual_machine_registration.ps1
    Write-Host 'Opengear Lighthouse Hyper-V Registration'
    Write-Host 'Creating Lighthouse HyperV Guest'

    $localServer = 1 # Indicate that we are connect to local or remote Hyper-V server
    $script_directory = Split-Path -parent $PSCommandPath

    # Check to make sure Ironman.vhd exists in the same folder as the executing script
    if (-not (Test-Path $script_directory\Ironman.vhd)) {
        Write-Warning "File Ironman.vhd is not found in '$script_directory'."
        pauseAndWait
        Exit 1
    }

    # Ask user which Hyper-V host Lighthouse will be connect to
    $hyperVHost = Read-Host -Prompt 'Enter Hyper-V host to deploy Lighthouse (default="localhost")'
    if ( -not $hyperVHost ) {
        $hyperVHost = $env:computername
    }

    Write-Host "Attempt to deploy Lighthouse in " $hyperVHost

    # Get the Hyper-V settings default folder for virtual hard disk so we can copy the disk there
    $VM_HOST = Get-VMHost -ComputerName $hyperVHost -ErrorAction SilentlyContinue -ErrorVariable errorGetVMHost
    if ($errorGetVMHost.Count -ne 0) {
        # check if error contain "You do not have the required permission to complete this task".
        # If so, as for credential and connect again
        if ($errorGetVMHost -like '*do not have the required permission to complete this task*' ) {
            $vmhostCredential = $Host.ui.PromptForCredential("Need credentials", "Please enter detail to connect to $hyperVHost.", "", "NetBiosUserName")
            if ( -not $vmhostCredential ) {
                Write-Warning "Credentials are needed to connect to $hyperVHost"
                pauseAndWait
                Exit 1
            }
            $VM_HOST = Get-VMHost -ComputerName $hyperVHost -Credential $vmhostCredential -ErrorAction SilentlyContinue -ErrorVariable errorGetVMHost
            if ($errorGetVMHost.Count -ne 0) {
                # This means remote manage Hyper-V has not been setup correctly, or credential was wrong
                Write-Warning "Failed to retrieve default HyperV Settings for $hyperVHost"
                Write-Warning "Error: " $errorGetVMHost
                pauseAndWait
                Exit 1
            }
            # We are now correct to remote hyper_v server
            $localServer = 0
        } else {
            # For any other error, display error message and exist
            Write-Warning "Failed to retrieve default HyperV Settings for $hyperVHost"
            Write-Warning "Error: " $errorGetVMHost
            pauseAndWait
            Exit 1
        }
    }

    if ( $env:ComputerName -ne $VM_HOST.ComputerName) {
        # Check to make sure specified $VM_HOST is not the local one
        $localServer = 0
    }

    $lh_name = Read-Host -Prompt 'Enter Lighthouse VM Name (default="Lighthouse")'
    if ( -not $lh_name ) {
        $lh_name = "Lighthouse"
    }
    # Check if name is already registered
    $registered_lhvm = Get-VM -CimSession $VM_HOST.CimSession | Sort-Object -Property Name | Select-Object -Property Name | Where-Object {$_.Name -match "$($lh_name)(_\d+)?$"} | Select-Object -Last 1
    if ($registered_lhvm) {
        # Add an incrementing digit to ensure uniqueness
        $lh_image_name = "$($lh_name)_" + ([int]($registered_lhvm.Name -replace $lh_name,'' -replace '\D','') + 1)
		Write-Warning "'$lh_name' is already registered. Using '$lh_image_name'."
    } else {
        $lh_image_name = $lh_name
    }

    # Check to make sure the $VM_HOST.VirtualHardDiskPath end with '\' If not add it
    if ($VM_HOST.VirtualHardDiskPath -notmatch '.+?\\$') {
        $lh_DestHardDisk = $VM_HOST.VirtualHardDiskPath + "\Opengear_$lh_image_name.vhd"
    } else {
        $lh_DestHardDisk = $VM_HOST.VirtualHardDiskPath + "Opengear_$lh_image_name.vhd"
    }

    if (Test-Path $lh_DestHardDisk) {
        Write-Warning "There is already an image called '$lh_DestHardDisk'."
        Write-Host "This is often caused by renaming or deleting VMs."
        Write-Host "If you renamed the VM, you can attempt to rename the image behind it."
        Write-Host "If you deleted the VM, you can delete the image."
        Write-Host "Otherwise, simply choose a different name."
        pauseAndWait
        Exit 1
    }

    # Copy VHD file to correct location
    if ($localServer -eq 1) {
        Write-Host "Copying $script_directory\Ironman.vhd to $lh_DestHardDisk"
        Copy-Item $script_directory\Ironman.vhd -Destination $lh_DestHardDisk
    } else {
        # create the remote session
        if ($vmhostCredential) {
            $destSession = New-PSSession -ComputerName $hyperVHost -Credential $vmhostCredential
        } else {
            $destSession = New-PSSession -ComputerName $hyperVHost
        }
        Write-Host "Copying $script_directory\Ironman.vhd to $lh_DestHardDisk on $hyperVHost"
        Copy-Item $script_directory\Ironman.vhd -Destination $lh_DestHardDisk -ToSession $destSession
    }

    # Get network switch with name specified above so it can be assigned the new VM. If not available then prompt the user to create one
    # Dont display red error when switch not found
    $test_switch_existed = Get-VMSwitch -CimSession $VM_HOST.CimSession $lh_NetworkSwitch -ErrorAction SilentlyContinue
    if ($test_switch_existed.Count -eq 0) {
        # Continuously prompt the user for a y/n response until we get one
        $prompt = "No virtual switch named '$lh_NetworkSwitch' detected. Create switch? [y/n]"
        $createPrivateSwitch = Read-Host $prompt
        while($createPrivateSwitch -ne "y" -and $createPrivateSwitch -ne "n")
        {
            $createPrivateSwitch = Read-Host $prompt
        }
        if ($createPrivateSwitch -eq "y") {
            Write-Host "Creating private network switch " $lh_NetworkSwitch " for " $lh_image_name ". Use the Virtual Switch Manager in Hyper-V Manager to edit its configuration."
            New-VMSwitch -CimSession $VM_HOST.CimSession -Name $lh_NetworkSwitch -SwitchType Private
        } else {
            Write-Host "Not creating a virtual switch for " $lh_image_name ". A switch can be added later via Hyper-V Manager."
        }
    }

    # Create the new Lighthouse image in Hyper-V
    Write-Host 'Creating new virtual machine:' $lh_image_name
    Write-Host 'Generation:' $lh_Generation
    Write-Host 'Memory:' ($lh_Base_Memory/1GB)'GB'
    Write-Host 'IDE boot disk:' $lh_DestHardDisk
    if ($test_switch_existed.Count -gt 0 -or $createPrivateSwitch -eq "y") {
        Write-Host 'Network Adapter:' $lh_NetworkSwitch
        New-VM -CimSession $VM_HOST.CimSession -VHDPath $lh_DestHardDisk -Name $lh_image_name -Generation $lh_Generation -MemoryStartupBytes $lh_Base_Memory -SwitchName $lh_NetworkSwitch
    } else {
        Write-Host 'Network Adapter:(not set)'
        New-VM -CimSession $VM_HOST.CimSession -VHDPath $lh_DestHardDisk -Name $lh_image_name -Generation $lh_Generation -MemoryStartupBytes $lh_Base_Memory
    }


    # Disable time-sync service Disable-VMIntegrationservice
    Disable-VMIntegrationService -CimSession $VM_HOST.CimSession -Name "Time Synchronization" -VMName $lh_image_name
    Write-Host 'Done'
    pauseAndWait
    Exit 0
} Catch {
    $ErrorMessage = $_.Exception.Message
    $FailedItem = $_.Exception.ItemName
    Write-Warning "Unexpected exception: " $ErrorMessage
    Write-Warning "Failed Item: " $FailedItem
    Write-Warning "Exception Type: $($_.Exception.GetType().FullName)" -ForegroundColor Red
    pauseAndWait
    Break
}
